<?php
/**
 * Eximus Ad Campaign Statistics Dashboard
 * @website: https://eximus.io
 */

include 'config.php';

// Read log file line by line to handle large files
$logs = [];
if (file_exists($LOG_FILE)) {
    $handle = fopen($LOG_FILE, 'r');
    if ($handle) {
        while (($line = fgets($handle)) !== false) {
            $line = rtrim($line, ",\n");
            if (!empty($line)) {
                $entry = json_decode($line, true);
                if ($entry) {
                    $logs[] = $entry;
                }
            }
        }
        fclose($handle);
    }
}

// Calculate statistics
$stats = [
    'total' => count($logs),
    'humans' => 0,
    'bots' => 0,
    'countries' => [],
    'cities' => [],
    'isps' => [],
    'recent' => []
];

foreach ($logs as $log) {
    // Count humans/bots
    if ($log['status'] === 'HUMAN') {
        $stats['humans']++;
    } else {
        $stats['bots']++;
    }
    
    // Country stats
    if (!empty($log['country_name'])) {
        $country = $log['country_name'] . ' (' . $log['country'] . ')';
        $stats['countries'][$country] = ($stats['countries'][$country] ?? 0) + 1;
    }
    
    // City stats
    if (!empty($log['city'])) {
        $stats['cities'][$log['city']] = ($stats['cities'][$log['city']] ?? 0) + 1;
    }
    
    // ISP stats
    if (!empty($log['isp'])) {
        $stats['isps'][$log['isp']] = ($stats['isps'][$log['isp']] ?? 0) + 1;
    }
}

// Get all logs in reverse order (latest first)
$stats['recent'] = array_reverse($logs);
$stats['unique_ips'] = count(array_unique(array_column($logs, 'ip')));

// Sort arrays by value
arsort($stats['countries']);
arsort($stats['cities']);
arsort($stats['isps']);

// Get top 5 for each
$stats['top_countries'] = array_slice($stats['countries'], 0, 5, true);
$stats['top_cities'] = array_slice($stats['cities'], 0, 5, true);
$stats['top_isps'] = array_slice($stats['isps'], 0, 5, true);

// Handle actions
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    if ($_GET['action'] === 'clear' && isset($_POST['confirm'])) {
        file_put_contents($LOG_FILE, '');
        echo json_encode(['success' => true]);
        exit;
    }
    
    if ($_GET['action'] === 'export') {
        // Remove request_uri from export
        $export_logs = array_map(function($log) {
            unset($log['request_uri']);
            return $log;
        }, $logs);
        header('Content-Disposition: attachment; filename="eximus_logs_' . date('Y-m-d') . '.json"');
        echo json_encode($export_logs, JSON_PRETTY_PRINT);
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Eximus AntiBot Dashboard</title>
    
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 15px 0;
        }
        
        .dashboard-header {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 12px 15px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            border-left: 4px solid;
            display: flex;
            align-items: center;
            height: 100%;
        }
        
        .stat-card.total {
            border-left-color: #667eea;
        }
        
        .stat-card.humans {
            border-left-color: #28a745;
        }
        
        .stat-card.bots {
            border-left-color: #dc3545;
        }
        
        .stat-card.unique {
            border-left-color: #17a2b8;
        }
        
        .stat-icon {
            font-size: 1.3rem;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            margin-right: 12px;
        }
        
        .stat-icon.total {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
        }
        
        .stat-icon.humans {
            background: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .stat-icon.bots {
            background: rgba(220, 53, 69, 0.1);
            color: #dc3545;
        }
        
        .stat-icon.unique {
            background: rgba(23, 162, 184, 0.1);
            color: #17a2b8;
        }
        
        .stat-content {
            flex: 1;
        }
        
        .stat-label {
            color: #6c757d;
            font-size: 0.75rem;
            margin-bottom: 2px;
        }
        
        .stat-number {
            font-size: 1.5rem;
            font-weight: bold;
            color: #333;
            line-height: 1;
        }
        
        .table-container {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
            max-height: 700px;
            overflow-y: auto;
        }
        
        .badge-human {
            background: #28a745;
            font-size: 0.75rem;
        }
        
        .badge-bot {
            background: #dc3545;
            font-size: 0.75rem;
        }
        
        .logo {
            font-size: 1.5rem;
            font-weight: bold;
            background: linear-gradient(135deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .live-indicator {
            display: inline-block;
            width: 8px;
            height: 8px;
            background: #28a745;
            border-radius: 50%;
            animation: pulse 2s infinite;
            margin-left: 8px;
        }
        
        @keyframes pulse {
            0% { box-shadow: 0 0 0 0 rgba(40, 167, 69, 0.7); }
            70% { box-shadow: 0 0 0 10px rgba(40, 167, 69, 0); }
            100% { box-shadow: 0 0 0 0 rgba(40, 167, 69, 0); }
        }
        
        .ip-badge {
            background: #f8f9fa;
            padding: 3px 8px;
            border-radius: 15px;
            font-family: monospace;
            font-size: 0.8rem;
        }
        
        .list-group-item {
            border: none;
            padding: 8px 12px;
            margin-bottom: 3px;
            background: #f8f9fa;
            border-radius: 8px;
            font-size: 0.85rem;
        }
        
        .geo-card {
            background: white;
            border-radius: 12px;
            padding: 15px;
            margin-bottom: 15px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
            max-height: 200px;
            overflow-y: auto;
        }
        
        .geo-card h6 {
            font-size: 0.9rem;
            margin-bottom: 10px;
            color: #667eea;
        }

        .table-sm td, .table-sm th {
            padding: 0.5rem;
            font-size: 0.85rem;
        }
        
        .progress {
            height: 4px;
            margin-top: 8px;
        }
        
        .stat-icon {
            font-size: 1.5rem;
            margin-bottom: 5px;
        }
    </style>
</head>
<body>
    <div class="container-fluid px-3">
        <!-- Header -->
        <div class="dashboard-header">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h1 class="logo mb-0">
                        <i class="bi bi-shield-check"></i> Eximus AntiBot
                        <span class="live-indicator"></span>
                    </h1>
                    <small class="text-muted">Advanced Traffic Protection | <a href="https://eximus.io" target="_blank">eximus.io</a></small>
                </div>
                <div class="col-md-6 text-md-end mt-2 mt-md-0">
                    <button class="btn btn-success btn-sm" onclick="location.reload()">
                        <i class="bi bi-arrow-clockwise"></i> Refresh
                    </button>
                    <button class="btn btn-primary btn-sm" onclick="exportLogs()">
                        <i class="bi bi-download"></i> Export
                    </button>
                    <button class="btn btn-danger btn-sm" onclick="clearLogs()">
                        <i class="bi bi-trash"></i> Clear
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Statistics Cards -->
        <div class="row">
            <div class="col-md-3">
                <div class="stat-card total">
                    <div class="stat-icon total">
                        <i class="bi bi-people-fill"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-label">Total Visitors</div>
                        <div class="stat-number"><?php echo number_format($stats['total']); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card humans">
                    <div class="stat-icon humans">
                        <i class="bi bi-person-check-fill"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-label">Humans Detected</div>
                        <div class="stat-number"><?php echo number_format($stats['humans']); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card bots">
                    <div class="stat-icon bots">
                        <i class="bi bi-robot"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-label">Bots Detected</div>
                        <div class="stat-number"><?php echo number_format($stats['bots']); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card unique">
                    <div class="stat-icon unique">
                        <i class="bi bi-globe"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-label">Unique IPs</div>
                        <div class="stat-number"><?php echo number_format($stats['unique_ips']); ?></div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Recent Visitors Table -->
        <div class="table-container mt-3">
            <h6 class="mb-3">
                <i class="bi bi-clock-history text-primary"></i> Recent Visitors
                <small class="text-muted float-end">Latest entries first</small>
            </h6>
            <table class="table table-hover table-sm">
                <thead class="table-light sticky-top">
                    <tr>
                        <th style="width: 8%">Time</th>
                        <th style="width: 12%">IP Address</th>
                        <th style="width: 15%">Location</th>
                        <th style="width: 12%">ISP/ASN</th>
                        <th style="width: 8%">Type</th>
                        <th style="width: 8%">HTTP Status</th>
                        <th style="width: 27%">User Agent</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($stats['recent'] as $log): ?>
                    <tr>
                        <td class="text-nowrap">
                            <?php echo date('H:i:s', strtotime($log['timestamp'])); ?><br>
                            <span class="text-muted"><?php echo date('Y-m-d', strtotime($log['timestamp'])); ?></span>
                        </td>
                        <td>
                            <span class="ip-badge"><?php echo htmlspecialchars($log['ip'] ?? 'unknown'); ?></span>
                        </td>
                        <td>
                            <strong><?php echo htmlspecialchars($log['city'] ?? 'Unknown'); ?></strong><br>
                            <?php echo htmlspecialchars($log['country_name'] ?? 'Unknown'); ?> 
                            <?php if (!empty($log['country'])): ?>(<?php echo htmlspecialchars($log['country']); ?>)<?php endif; ?>
                            <?php if (!empty($log['timezone'])): ?>
                            <br><span class="text-muted">Timezone: <?php echo htmlspecialchars($log['timezone']); ?></span>
                            <?php endif; ?>
                        </td>
                        <td class="text-nowrap">
                            <?php echo htmlspecialchars($log['isp'] ?? $log['org'] ?? 'Unknown'); ?><br>
                            <span class="text-muted">ASN: <?php echo htmlspecialchars($log['asn'] ?? 'N/A'); ?></span>
                        </td>
                        <td>
                            <?php if ($log['status'] === 'HUMAN'): ?>
                                <span class="badge badge-human">
                                    <i class="bi bi-person-check"></i> Human
                                </span>
                            <?php else: ?>
                                <span class="badge badge-bot">
                                    <i class="bi bi-robot"></i> Bot
                                </span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php 
                            $action = $log['action'] ?? '';
                            $statusClass = '';
                            $statusText = '';
                            
                            switch($action) {
                                case '200':
                                    $statusClass = 'text-success';
                                    $statusText = '200 OK';
                                    break;
                                case '302':
                                    $statusClass = 'text-info';
                                    $statusText = '302 Redirect';
                                    break;
                                case '403':
                                    $statusClass = 'text-danger';
                                    $statusText = '403 Forbidden';
                                    break;
                                case '404':
                                    $statusClass = 'text-warning';
                                    $statusText = '404 Not Found';
                                    break;
                                default:
                                    $statusClass = 'text-muted';
                                    $statusText = $action ?: 'N/A';
                            }
                            ?>
                            <span class="<?php echo $statusClass; ?> fw-bold">
                                <?php echo htmlspecialchars($statusText); ?>
                            </span>
                        </td>
                        <td>
                            <span class="clickable-ua text-truncate d-inline-block text-wrap">
                                <?php echo htmlspecialchars($log['user_agent'] ?? 'Unknown'); ?>
                            </span>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($stats['recent'])): ?>
                    <tr>
                        <td colspan="7" class="text-center text-muted py-4">
                            <i class="bi bi-inbox" style="font-size: 2rem;"></i>
                            <p class="mt-2">No visitor logs yet</p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>

        // Clear logs function
        function clearLogs() {
            if (confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
                fetch('?action=clear', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'confirm=1'
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Logs cleared successfully!');
                        location.reload();
                    }
                });
            }
        }
        
        // Export logs function
        function exportLogs() {
            window.location.href = '?action=export';
        }
        
        // Auto-refresh every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html>